<?php

namespace App\Http\Controllers\Videos;

use App\Http\Controllers\Controller;
use App\Http\Resources\VideoResource;
use App\Http\Traits\GeneralTrait;
use App\Models\TradingType;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class VideoController extends Controller
{
    use GeneralTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'video_url' => 'required|string',
            'uuidTradingType' => 'required|exists:trading_types,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {

            $tradingType = TradingType::where('uuid', $request->uuidTradingType)->firstOrFail();

            $video = Video::create([
                'uuid' => Str::uuid(),
                'video_url' => $request->input('video_url'),
                'trading_type_id' => $tradingType->id,
                'status' =>  1
            ]);

            $data['video'] = new VideoResource($video);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Video  $video
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidVideo' => 'required|string|exists:videos,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $video = Video::where('uuid', $request->uuidVideo)->firstOrFail();
            $data['video']=new VideoResource($video);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Video  $video
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'video_url' => 'string',
            'uuidVideo' => 'required|string|exists:videos,uuid',
            'uuidTradingType' => 'required|exists:trading_types,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {

            $tradingType = TradingType::withoutGlobalScope('active')->where('uuid', $request->uuidTradingType)->firstOrFail();

            $video = Video::where('uuid', $request->uuidVideo)->firstOrFail();

            $video->update([
                'video_url' => $request->input('video_url', $video->video_url),
                'trading_type_id' => $tradingType->id,
            ]);
            $data['video'] = new VideoResource($video);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function deactivateVideo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidVideo' => 'required|string|exists:videos,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {
            $video = Video::where('uuid', $request->uuidVideo)->firstOrFail();
            $video->update(['status' => 0]);

            $data['message'] = 'The video is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function getVideosByTradingType($uuidTradingType)
    {
        $validator = Validator::make(['uuidTradingType' => $uuidTradingType], [
            'uuidTradingType' => 'required|string|exists:trading_types,uuid',

        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {
            $tradingType = TradingType::where('uuid', $uuidTradingType)->first();

            if (!$tradingType) {
                return $this->notFoundResponse('Trading type not found');
            }

            $videos = $tradingType->videos;

            $data['videos'] = VideoResource::collection($videos);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function activateVideo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidVideo' => 'required|string'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $video = Video::withoutGlobalScope('active')->where('uuid', $request->uuidVideo)->firstOrFail();

            if ($video->status == 1) {
                return $this->apiResponse(null, false, 'The video is already active.', 400);
            }

            $video->update(['status' => 1]);

            $data['message'] = 'The video is now active.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function deleteVideo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidVideo' => 'required|string|exists:videos,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $video = Video::withoutGlobalScope('active')->where('uuid', $request->uuidVideo)->firstOrFail();
            $video->delete();

            $data['message'] = 'The video has been deleted.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


}
