<?php

namespace App\Http\Controllers\Users;

use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource;
use App\Http\Traits\FileUploader;
use App\Http\Traits\GeneralTrait;
use App\Models\Renewal;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    use GeneralTrait;
    use FileUploader;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            // Retrieve all users, including inactive ones
            $users = User::withoutGlobalScope('active')->get();

            $data['users'] = UserResource::collection($users);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function show(Request $request)
    {
        // Validate the incoming request data
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {

            // Retrieve the user with the given UUID and status
            $user = User::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();


            $data['user'] = new UserResource($user);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function accountUser()
    {
        try {
            $user = auth('sanctum')->user();

            if (!$user) {
                return $this->unAuthorizeResponse();
            }

            $data['user'] = new UserResource($user);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }



    public function updateUser(Request $request)
    {
        // Validate the incoming request data
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'address' => 'required|string',
            'zip' => 'required|digits:4',
            'city' => 'required|string',
            'phone' => 'required|string',
        ]);

        // Check if validation fails and return errors if any
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            // Get the authenticated user
            $user = auth('sanctum')->user();
            if (!$user) {
                return $this->unAuthorizeResponse();
            }

            // Update user data with the request data
            $user->update([
                'first_name' => $request->input('first_name'),
                'last_name' => $request->input('last_name'),
                'address' => $request->input('address'),
                'zip' => $request->input('zip'),
                'city' => $request->input('city'),
                'phone' => $request->input('phone'),
            ]);

            // Generate a token for the updated user
            $data['user'] = new UserResource($user);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }




    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'old_password' => 'required|string',
            'new_password' => 'required|string|min:8|confirmed',

        ]);

        // Check if validation fails and return errors if any
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return $this->unAuthorizeResponse();
            }
            if (!Hash::check($request->old_password, $user->password)) {
                return $this->requiredField('Invalid old password');
            }

            $user->update([
                 'password' => Hash::make($request->new_password),
             ]);
            $data['user'] = new UserResource($user);

            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }

    }


    public function updatePhotoProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'url_image' => 'required|image'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return $this->unAuthorizeResponse();
            }
            $url = $this->uploadImagePrivate($request, $user, 'profile', 'url_image');
            $user->update(['url_image' => $url,]);

            $data['user'] = new UserResource($user);

            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }

    }

    public function deactivateUser()
    {
        try {
            $user = auth('sanctum')->user();

            if (!$user) {
                return $this->unAuthorizeResponse();
            }

            $user->update(['status' => 0,
                         'email_verified_at' => null,
                         'email_status' => 0]);

            $data['message'] = 'The operation was successful, and the user is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function activeUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid_user' => 'required|string'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $user = User::withoutGlobalScope('active')
                ->where('uuid', $request->uuid_user)
                ->where('email_status', 1)
                ->first();

            if(!$user) {
                return $this->requiredField('User not verified via email');

            }

            // Update user status
            $user->update([
                'status' => 1,
                'date_active' => Carbon::now(), // Set to the current date
                'date_renewal' => Carbon::now()->addMonth(), // Add one month to the current date
            ]);

            $data['user'] = new UserResource($user);
            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function deactivateClient(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'uuid_user' => 'required|string'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $user = User::where('uuid', $request->uuid_user)
                ->firstOrFail();


            $user->update(['status' => 0,
                         'email_verified_at' => null,
                         'email_status' => 0]);

            $data['message'] = 'The operation was successful, and the user is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }




    public function renewalDateExtension(Request $request)
    {
        try {
            $admin = Auth::guard('admin')->user();

            if (!$admin) {
                return $this->unAuthorizeResponse();
            }

            $validator = Validator::make($request->all(), [
                'uuid_user' => 'required|string'
            ]);

            if ($validator->fails()) {
                return $this->requiredField($validator->errors()->first());
            }
            $user = User::withoutGlobalScope('active')
            ->where('uuid', $request->uuid_user)
            ->where('email_status', 1)
            ->firstOrFail();

            // Create a renewal record for the admin
            $renewal = Renewal::create([
                'uuid' => Str::uuid(),
                'admin_id' => $admin->id,
                'type' => 'cash',
                'user_id' => $user->id,
            ]);

            // Retrieve the user to update based on the provided UUID


            // Update user status and date values
            $user->update([
                'status' => 1,
                'date_active' => Carbon::now(),
                'date_renewal' => Carbon::now()->addMonth(),
            ]);

            $data['renewal'] = $renewal;
            $data['user'] = new UserResource($user);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function emailStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid_user' => 'required|string'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $user = User::withoutGlobalScope('active')
            ->where('uuid', $request->uuid_user)
            ->first();

            if(!$user) {
                return $this->requiredField('User not verified via email');

            }

            // Update user status
            $user->update([
                'email_status' => 1,
            ]);

            $data['user'] = new UserResource($user);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }
}
