<?php

namespace App\Http\Controllers\TradingTypes;

use App\Http\Controllers\Controller;
use App\Http\Resources\PairResource;
use App\Http\Resources\TradingTypeResource;
use App\Http\Resources\VideoResource;
use App\Http\Traits\GeneralTrait;
use App\Models\TradingType;
use App\Models\Video;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class TradingTypeController extends Controller
{
    use GeneralTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            // Temporarily remove the global scope to get all TradingType records
            $tradingType = TradingType::withoutGlobalScope('active')->get();

            $data['tradingType'] = TradingTypeResource::collection($tradingType);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name_type' => 'required|string|unique:trading_types,name_type',
            'video_name' => 'nullable|string',
            'video_status' => 'nullable|boolean',
            'chart_name' => 'nullable|string',
            'chart_status' => 'nullable|boolean',
        ]);
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {

            $tradingType  = TradingType::create([
                'uuid' => Str::uuid(),
                'name_type' => $request->input('name_type'),
                'video_status' => $request->video_status,
                'chart_status' => $request->chart_status,
                'video_name' => $request->video_name,
                'chart_name' => $request->chart_name,
            ]);
            $data['tradingType'] = new TradingTypeResource($tradingType);

            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\TradingType  $tradingType
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid_trading_type' => 'required|string|exists:trading_types,uuid'
        ]);
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {
            $tradingType = TradingType::withoutGlobalScope('active')->where('uuid', $request->uuid_trading_type)->firstOrFail();

          $videos = Video::withoutGlobalScope('active')->where('trading_type_id', $tradingType->id)->get();

            $data['tradingType'] = new TradingTypeResource($tradingType);
            $data['pairs'] = PairResource::collection($tradingType->pairs);
            $data['video']=VideoResource::collection($videos);

            return $this->apiResponse($data, true, null, 200);


        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }

    }


    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name_type' => [
                'required',
                'string',
                Rule::unique('trading_types', 'name_type')
                    ->where('uuid', $request->uuid)
                    ->ignore($request->uuid, 'uuid')],
            'uuid' => 'required|string|exists:trading_types,uuid',
            'video_name' => 'nullable|string',
            'video_status' => 'nullable|boolean',
            'chart_name' => 'nullable|string',
            'chart_status' => 'nullable|boolean',

        ]);
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {

            $tradingType = TradingType::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();

            $tradingType->update([
                 'name_type' => $request->input('name_type'),
                 'video_status' => $request->video_status,
                 'chart_status' => $request->chart_status,
                 'video_name' => $request->video_name,
                 'chart_name' => $request->chart_name,
             ]);
            $data['tradingType'] = new TradingTypeResource($tradingType);

            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function deactivateTradingType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:trading_types,uuid'
        ]);
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {


            $tradingType = TradingType::where('uuid', $request->input('uuid'))->firstOrFail();
            $tradingType->update(['status' => 0]);

            $data['message'] = 'The operation was successful, and the tradingType is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function activateTradingType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            // Temporarily remove the global scope
          $tradingType=TradingType::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();

            $tradingType->update(['status' => 1]);

            $data['message'] = 'The operation was successful, and the tradingType is now active.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function allTradingType()
    {
        try {
            $tradingType = TradingType::all();

            $data['tradingType'] = TradingTypeResource::collection($tradingType);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

}
