<?php

namespace App\Http\Controllers\Pairs;

use App\Http\Controllers\Controller;
use App\Http\Resources\PairResource;
use App\Http\Resources\PairVariableResource;
use App\Http\Resources\PairWithVariablesResource;
use App\Http\Traits\FileUploader;
use App\Http\Traits\GeneralTrait;
use App\Models\Favorite;
use App\Models\Notification;
use App\Models\Pair;
use App\Models\PairVariable;
use App\Models\TradingType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class PairController extends Controller
{
    use GeneralTrait;
    use FileUploader;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $pairs = Pair::withoutGlobalScope('active')->get();

            $data['pairs'] = PairResource::collection($pairs);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidTradingType' => 'required|exists:trading_types,uuid',
            'name_pair' => 'required|string',
            'icon_url' => 'image',
            'text_notification' => 'string'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $tradingType = TradingType::where('uuid', $request->uuidTradingType)->firstOrFail();


            $pair = Pair::create([
                'uuid' => Str::uuid(),
                'trading_type_id' => $tradingType->id,
                'name_pair' => $request->input('name_pair'),
                'status' => 1 ,
            ]);

            if ($request->hasFile('icon_url')) {
                $url = $this->uploadImagePublic($request, $pair, 'Pairs_icon', 'icon_url');
                $pair->update(['icon_url' => $url]);
            }
            Notification::create([
                'uuid' => Str::uuid(),
                'text_notification' => $request->text_notification,
                'trading_type_id' => $pair->trading_type_id,
                'pair_id' => $pair->id,
                'status' => 1 ,
            ]);

            $data['pair'] = new PairResource($pair);

            return $this->apiResponse($data, true, null, 201);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Pair  $pair
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidPair' => 'required|string|exists:pairs,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pair = Pair::withoutGlobalScope('active')->where('uuid', $request->uuidPair)->firstOrFail();
            $pairVariables=PairVariable::withoutGlobalScope('active')->where('pair_id',$pair->id)->get();

            $data['pair'] = new PairResource($pair);
            $data['pair_variables']=PairVariableResource::collection($pairVariables);


            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Pair  $pair
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidPair' => 'required|string|exists:pairs,uuid',
            'uuidTradingType' => 'required|exists:trading_types,uuid',
            'name_pair' => 'required|string',
            'icon_url' => 'image',
            'text_notification' => 'string'

        ]);
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {

            $pair = Pair::withoutGlobalScope('active')->where('uuid', $request->input('uuidPair'))->firstOrFail();

            $tradingType = TradingType::where('uuid', $request->uuidTradingType)
            ->where('status', 1)
            ->firstOrFail();

            $pair->update([
              'trading_type_id' => $tradingType->id,
              'name_pair' => $request->input('name_pair'),
            ]);
            if ($request->hasFile('icon_url')) {
                $url = $this->uploadImagePublic($request, $pair, 'Pairs_icon', 'icon_url');
                $pair->update(['icon_url' => $url]);
            }
            if($request->text_notification){
                $notification=$pair->notification()->first();
                $notification->update(
                    [
                        'text_notification' => $request->text_notification,
                        'trading_type_id' => $pair->trading_type_id,
                    ]
                    );
            }

            $data['pair'] = new PairResource($pair);

            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Pair  $pair
     * @return \Illuminate\Http\Response
     */
    public function deactivatePair(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:pairs,uuid'
        ]);
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {
            $pairUUID = $request->input('uuid');
            $allFavorites = Favorite::all();
            // $allFavorites=$allFavorites->pair_favorite;

            foreach ($allFavorites as $favorite) {
                $currentPairs = json_decode($favorite->pair_favorite, true) ?? [];


                $currentPairs = array_filter($currentPairs, function ($pair) use ($pairUUID) {
                    return $pair['pair_uuid'] !== $pairUUID;
                });

                $favorite->pair_favorite = json_encode($currentPairs);
                $favorite->save();
                if($currentPairs){
                    $favorite->delete();
                }
            }


            $pairs = Pair::where('uuid', $request->input('uuid'))->firstOrFail();
            $pairs->update(['status' => 0]);

            $data['message'] = 'The pair is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function activatePair(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:pairs,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pair = Pair::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();

            // Check if the pair is already active
            if ($pair->status == 1) {
                return $this->apiResponse(null, false, 'The pair is already active.', 400);
            }

            $pair->update(['status' => 1]);

            $data['message'] = 'The pair is now active.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function getPairsByTradingType($uuid_trading_type)
    {

        $validator = Validator::make(['uuid_trading_type' => $uuid_trading_type], [
            'uuid_trading_type' => 'required|string|exists:trading_types,uuid',

        ]);

        // Check if validation fails and return errors if any
        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }
        try {
            // Find the trading type by its UUID
            $tradingType = TradingType::where('uuid', $uuid_trading_type)->first();

            // Check if the trading type exists
            if (!$tradingType) {
                return $this->notFoundResponse('Trading type not found');
            }

            $pairs = $tradingType->pairs()->where('status', 1)->get();

            if (!$pairs) {
                return $this->notFoundResponse('No active pairs found for the trading type');
            }

            // Format the pairs data using the PairResource
            $data['tradingType'] = $tradingType->name_type;
            $data['pairs'] = PairResource::collection($pairs);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            // Handle and log errors
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function getPairsWithVariablesByTradingType($uuid_trading_type)
    {
        $validator = Validator::make(['uuid_trading_type' => $uuid_trading_type], [
            'uuid_trading_type' => 'required|string|exists:trading_types,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {

            $tradingType = TradingType::where('uuid', $uuid_trading_type)->firstOrFail();




            $pairs = $tradingType->pairs()->where('status', 1)->get();


            $data['pairs'] = PairWithVariablesResource::collection($pairs);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {

            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function getPairWithVariables($uuidPair)
    {
        $validator = Validator::make(['uuidPair' => $uuidPair], [
            'uuidPair' => 'required|string|exists:pairs,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {

            $pair = Pair::where('uuid', $uuidPair)->where('status', 1)->first();

            if (!$pair) {
                return $this->notFoundResponse('Pair not found or not active');
            }

            $pairVariables = $pair->pairVariable;
            $data['pairWithVariable'] = new PairWithVariablesResource($pair);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

}
