<?php

namespace App\Http\Controllers\PairVariables;

use App\Http\Controllers\Controller;
use App\Http\Resources\PairVariableResource;
use App\Http\Traits\FileUploader;
use App\Http\Traits\GeneralTrait;
use App\Models\Pair;
use App\Models\PairVariable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class PairVariableController extends Controller
{
    use GeneralTrait ;
    use FileUploader;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string',
            'image_url' => 'image',
            'text_description' => 'required|string',
            'icon_url' => 'image',
            'pair_uuid' => 'required|exists:pairs,uuid',
            'trading_note' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pair = Pair::where('uuid', $request->pair_uuid)->where('status', 1)->firstOrFail();

            if (!$pair) {
                return $this->notFoundResponse('Pair not found or not active');
            }
            $pairVariable = PairVariable::create([
                'uuid' => Str::uuid(),
                'title' => $request->input('title'),
                'text_description' => $request->input('text_description'),
                'pair_id' => $pair->id,
                'trading_note' => $request->trading_note,
                'status' => 1,
            ]);

            if ($request->hasFile('icon_url')) {
                $url = $this->uploadImagePublic($request, $pair, 'PairVariable_icon', 'icon_url');
                $pairVariable->update(['icon_url' => $url]);
            }
            if ($request->hasFile('image_url')) {
                $urlImage = $this->uploadImagePublic($request, $pair, 'PairVariable_image', 'image_url');
                $pairVariable->update(['image_url' => $urlImage]);
            }

            $data['pairVariable'] = new PairVariableResource($pairVariable);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }




    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidPairVariable' => 'required|string|exists:pair_variables,uuid,status,1'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pairVariable = PairVariable::withoutGlobalScope('active')->where('uuid', $request->uuidPairVariable)->firstOrFail();
            $data['pairVariable'] = new PairVariableResource($pairVariable);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuidPairVariable' => 'required|string|exists:pair_variables,uuid,status,1',
            'title' => 'required|string',
            'image_url' => 'image',
            'text_description' => 'required|string',
            'icon_url' => 'image',
            'pair_uuid' => 'required|exists:pairs,uuid',
            'trading_note' => 'nullable|string'

        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pairVariable = PairVariable::withoutGlobalScope('active')->where('uuid', $request->input('uuidPairVariable'))->firstOrFail();
            $pair = Pair::where('uuid', $request->pair_uuid)->where('status', 1)->firstOrFail();

            if (!$pair) {
                return $this->notFoundResponse('Pair not found or not active');
            }

            $pairVariable->update([
                'title' => $request->input('title'),
                'text_description' => $request->input('text_description'),
                'pair_id' => $pair->id,
                'trading_note' => $request->trading_note,
            ]);
            if ($request->hasFile('icon_url')) {
                $url = $this->uploadImagePublic($request, $pair, 'PairVariable_icon', 'icon_url');
                $pairVariable->update(['icon_url' => $url]);
            }
            if ($request->hasFile('image_url')) {
                $urlImage = $this->uploadImagePublic($request, $pair, 'PairVariable_image', 'image_url');
                $pairVariable->update(['image_url' => $urlImage]);
            }

            $data['pairVariable'] = new PairVariableResource($pairVariable);

            return $this->apiResponse($data, true, null, 200);

        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function deactivatePairVariable(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:pair_variables,uuid,status,1'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pairVariable = PairVariable::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();
            $pairVariable->update(['status' => 0]);

            $data['message'] = 'The pair variable is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function activatePairVariable(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:pair_variables,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pairVariable = PairVariable::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();

            // Check if the pair variable is already active
            if ($pairVariable->status == 1) {
                return $this->apiResponse(null, false, 'The pair variable is already active.', 400);
            }

            $pairVariable->update(['status' => 1]);

            $data['message'] = 'The pair variable is now active.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function deletePairVariable(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:pair_variables,uuid'
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $pairVariable = PairVariable::withoutGlobalScope('active')->where('uuid', $request->input('uuid'))->firstOrFail();
            $pairVariable->delete();

            $data['message'] = 'The pair variable has been deleted.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


}
