<?php

namespace App\Http\Controllers\Companies;

use App\Http\Controllers\Controller;
use App\Http\Resources\CompanyResource;
use App\Http\Traits\GeneralTrait;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CompanyController extends Controller
{
    use GeneralTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            $companies = Company::withoutGlobalScope('active')->get();

            $data['companies'] = CompanyResource::collection($companies);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        $name = trim($request->input('name'));
        $name = preg_replace('/\s+/', '_', $name);

        $code = Str::random(12);

        try {
            $company = Company::create([
                'uuid' => Str::uuid(),
                'name' => $name,
                'code' => $code,
                'status' => 1,
            ]);

            $data['company'] = new CompanyResource($company);

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {

            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Company  $company
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:companies,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $company =  Company::withoutGlobalScope('active')->where('uuid', $request->uuid)->firstOrFail();
            $data['company'] =new CompanyResource($company);
            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }
    public function update(Request $request)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|string',
        'uuid' => 'required|string|exists:companies,uuid'
    ]);

    if ($validator->fails()) {
        return $this->requiredField($validator->errors()->first());
    }

    try {
        $company = Company::withoutGlobalScope('active')->where('uuid', $request->uuid)->firstOrFail();

        $name = trim($request->input('name'));
        $name = preg_replace('/\s+/', '_', $name);

        $company->update([
            'name' => $name,
        ]);

        $data['company'] = new CompanyResource($company);

        return $this->apiResponse($data, true, null, 200);
    } catch (\Exception $ex) {
        return $this->apiResponse(null, false, $ex->getMessage(), 500);
    }
}


    public function activate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:companies,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $company = Company::withoutGlobalScope('active')->where('uuid', $request->uuid)->firstOrFail();
            $company->update(['status' => 1]);

            $data['message'] = 'The company is now active.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }

    public function deactivate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'uuid' => 'required|string|exists:companies,uuid',
        ]);

        if ($validator->fails()) {
            return $this->requiredField($validator->errors()->first());
        }

        try {
            $company = Company::where('uuid', $request->uuid)->firstOrFail();
            $company->update(['status' => 0]);

            $data['message'] = 'The company is now inactive.';

            return $this->apiResponse($data, true, null, 200);
        } catch (\Exception $ex) {
            return $this->apiResponse(null, false, $ex->getMessage(), 500);
        }
    }


    public function destroy(Request $request)
{
    $validator = Validator::make($request->all(), [
        'uuid' => 'required|string|exists:companies,uuid',
    ]);

    if ($validator->fails()) {
        return $this->requiredField($validator->errors()->first());
    }

    try {
        $company = Company::withoutGlobalScope('active')->where('uuid', $request->uuid)->firstOrFail();

        // Soft delete the company
        $company->delete();

        $data['message'] = 'The company has been deleted successfully.';

        return $this->apiResponse($data, true, null, 200);
    } catch (\Exception $ex) {
        return $this->apiResponse(null, false, $ex->getMessage(), 500);
    }
}

}
